/*
 *  Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 *  Copyright (C) 2008-2008 - INRIA - Allan SIMON
 *
 *  This file must be used under the terms of the CeCILL.
 *  This source file is licensed as described in the file COPYING, which
 *  you should have received as part of this distribution.  The terms
 *  are also available at
 *  http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 *
 */

#include <assert.h>
#include <stdio.h>
#include "matrixExponential.h" 


#define LEADDIM 10

#define DMATRIX_IN  {0.15869047446176410,0.80895681912079453,0.72621259605512023,0.18993748771026731,\
 0.43964602379128337,0.37921421090140939,0.95195201179012656,0.59145097108557820,\
 0.70597065938636661,0.62873698258772492,\
0.62407150492072105,0.68317985069006681,0.70999773032963276,0.25839814636856318,\
 0.65407369285821915,0.76687160786241293,0.71278580836951733,0.68067426700145006,\
 0.70181696489453316,0.28785153012722731,\
0.63760355720296502,0.03401931514963508,0.47457459988072515,0.09878737432882190,\
 0.58781063789501786,0.60066213225945830,0.11923700617626309,0.07392961112782359,\
 0.40879997471347451,0.3292048736475408,\
0.42704886104911566,0.23805456422269344,0.9438692079856992,0.06199027225375175,\
 0.60208318661898375,0.78567356057465076,0.50091631803661585,0.94336947053670883,\
 0.06362213846296072,0.47192330472171307,\
0.10716815060004592,0.94920115964487195,0.14596485672518611,0.04034968325868249,\
 0.04535020282492042,0.73871155502274632,0.32900535268709064,0.12863306934013963,\
 0.06573933875188231,0.33537696348503232,\
0.23822965659201145,0.21827886346727610,0.07141010463237762,0.74001471605151892,\
 0.20294443331658840,0.55442603398114443,0.48089468106627464,0.20190807711333036,\
 0.53310041315853596,0.55530697200447321,\
0.94629473658278584,0.61546878470107913,0.67337385797873139,0.61626600986346602,\
 0.78442738251760602,0.99291495559737086,0.33036959776654840,0.19693034281954169,\
 0.03315818728879094,0.11960808141157031,\
0.45766852516680956,0.83135433867573738,0.65369247179478407,0.6583583448082209,\
 0.26375361625105143,0.97574284672737122,0.63044753577560186,0.89286901615560055,\
 0.31578356493264437,0.76139996573328972,\
0.89644787041470408,0.77340125897899270,0.19968961318954825,0.25145971449092031,\
 0.43832763982936740,0.37096222722902894,0.21171907847747207,0.46179189579561353,\
 0.37858232436701655,0.47909884760156274,\
0.44384705275297165,0.4244190966710448,0.60141251794993877,0.38433500844985247,\
 0.86648589745163918,0.30322382133454084,0.44860231317579746,0.62512917164713144,\
 0.46195234358310699,0.28169692959636450}

#define DMATRIX_RESULT {14.6458099031118518,15.9879935334969154,14.5023146230340352,9.79104027026660262,\
 13.658147977601745,18.2826677794023276,13.9913488019182726,13.0555850244756648,\
 11.2702097163264696,12.2080852641231488,\
14.9576657268129818,18.0415870145381838,15.4577532112657572,10.6060207869925502,\
 14.6987916130120198,19.9917861348452170,14.8484467294549827,14.0994958596124,\
 12.1294313366477713,12.8836800651605152,\
7.72729140087719646,8.27571528291814396,8.78222511899259928,5.22537042408334251,\
 7.61718988429527233,10.1106593547443495,7.22727291477849576,6.72646688774223733,\
 6.21430065514425589,6.69442204331610213,\
11.8618893865302528,13.2731189359057637,12.8373872935372706,9.52692469532374631,\
 11.9056327368823247,16.3654355264039921,11.8674243780595265,11.7206072002431814,\
 9.24712051368910970,10.6966525674175159,\
7.25089971320127891,8.98673751651268127,7.49831172728240691,5.29368504368777693,\
 8.12154956830077346,10.3089262075282218,7.4337348502224856,6.83888206810676191,\
 5.87125977419815737,6.5321378683758375,\
9.25854069422946502,10.4059421461299539,9.47515410794143342,7.263789985253827,\
 9.16034813421007321,13.6353552693330933,9.3726649654333336,8.83558176180042132,\
 7.6055387753197277,8.4871270614444292,\
12.6878667266786511,14.1417583108624711,13.0448499416136254,9.182764814582713,\
 12.4852121681423256,17.073953108953809,13.2550267101027224,11.3640159258193041,\
 9.6279844249894015,10.6765282334921068,\
16.0691497196859565,18.4667994666424704,16.975871456346809,12.083598083627443,\
 15.7522021028722747,22.0489520587189212,16.1322911282790358,16.8044904865300744,\
 12.7829232118433556,14.5730443462997599,\
11.943978463537455,13.6259118560125358,11.8068134298221956,8.21687415305816771,\
 11.4127920042158735,15.2296090993128281,11.3826577511842082,11.0976684273648587,\
 10.3857413785092909,10.2894980707345525,\
11.5843985401712821,13.2781579371486753,12.1725305229437293,8.37850128011447204,\
 11.8141382209783163,15.4022074994757734,11.484608766021573,11.1535660239869383,\
 9.31343988645433285,11.1349864749012912}





#define ZRMATRIX_IN {0.21132486546412110,0.56084860628470778,0.30760907428339124,0.50153415976092219, \
 0.28064980218186975,0.40948254754766822,0.38737787725403905,0.53762298030778766,\
 0.58787201577797532,0.64885628735646605,\
0.75604385416954756,0.66235693730413914,0.93296162132173777,0.43685875833034515,\
 0.12800584640353918,0.87841258011758327,0.92228986788541079,0.11999255046248436,\
 0.48291792999953032,0.99231909401714802,\
0.00022113462910056,0.72635067673400044,0.21460078610107303,0.26931248093023896,\
 0.77831285959109664,0.11383596854284406,0.94881842611357570,0.2256303490139544,\
 0.22328650346025825,0.05004197778180242,\
0.33032709173858166,0.19851438421756029,0.31264199689030647,0.63257448654621840,\
 0.21190304495394230,0.19983377400785685,0.34353372454643250,0.62740930821746588,\
 0.84008856676518917,0.74855065811425447,\
0.66538110421970487,0.54425731627270579,0.36163610080257058,0.40519540151581168,\
 0.11213546665385365,0.56186607433483005,0.37601187312975526,0.76084325974807143,\
 0.12059959070757031,0.41040589986369014,\
0.62839178834110498,0.23207478970289230,0.2922266637906432,0.91847078315913677,\
 0.68568959552794695,0.58961773291230202,0.73409405630081892,0.04855662025511265,\
 0.28553641680628061,0.60845263302326202,\
0.84974523587152362,0.23122371966019273,0.56642488157376647,0.04373343335464597,\
 0.15312166837975383,0.68539796629920602,0.26157614728435874,0.67239497276023030,\
 0.86075146449729800,0.85442108893766999,\
0.68573101982474327,0.21646326314657927,0.48264719732105732,0.48185089323669672,\
 0.69708506017923355,0.89062247332185507,0.49934938363730907,0.20171726960688829,\
 0.84941016510128975,0.06426467280834913,\
0.87821648130193353,0.88338878145441413,0.33217189135029912,0.26395560009405017,\
 0.84155184263363481,0.50422128057107329,0.26385784195736051,0.39115739194676280,\
 0.52570608118548989,0.82790829380974174,\
0.06837403681129217,0.65251349471509457,0.59350947011262178,0.41481037065386772,\
 0.40620247554033995,0.34936154074966908,0.52535630855709314,0.83003165572881699,\
 0.99312098976224661,0.92623437754809856}

#define ZIMATRIX_IN {0.56672112690284848,0.73956565884873271,0.02587099233642220,0.70648676296696067,\
 0.96770532103255391,0.42568723810836673,0.18711117887869477,0.21240556007251143,\
 0.69127879338338971,0.13049928424879909,\
0.57116389367729425,0.00371731072664261,0.51744682248681784,0.52114724926650524,\
 0.50685344357043505,0.24615605548024178,0.01895748358219862,0.57950199581682682,\
 0.76568587962538004,0.96820035576820374,\
0.81601104838773608,0.59005728596821427,0.39168732380494475,0.28704008506610990,\
 0.52329764096066356,0.92295324662700295,0.84335647663101554,0.26281475694850087,\
 0.35726496530696750,0.6561381467618048,\
0.05689279362559319,0.3096467358991504,0.24135384149849415,0.65027950797230005,\
 0.5596947595477104,0.10007458087056875,0.07485948316752911,0.43609866220504045,\
 0.76933998242020607,0.24455389659851789,\
0.55959366867318749,0.25522057106718421,0.50644348980858922,0.08813347620889544,\
 0.56173070007935166,0.46782181179150939,0.85328151332214475,0.91105451015755534,\
 0.54776339093223214,0.52831235667690635,\
0.12493403162807226,0.62518793903291225,0.42361020017415285,0.44987633451819420,\
 0.46817600261420012,0.39504976756870747,0.01245901081711054,0.80826673656702042,\
 0.09622887428849936,0.84689256362617016,\
0.72792222863063216,0.11574173765257001,0.28937275288626552,0.72272530803456903,\
 0.77945467224344611,0.03661171020939946,0.18675393564626575,0.81026530498638749,\
 0.95611717319115996,0.78766220854595304,\
0.26777664758265018,0.61170040909200907,0.08879321813583374,0.89767962601035833,\
 0.79010718129575253,0.51753685344010592,0.49205840751528740,0.25904283951967955,\
 0.22074085660278797,0.12620826158672571,\
0.54653349192813039,0.67839562846347690,0.62128817522898316,0.24278218811377883,\
 0.98085420625284314,0.83254515705630183,0.74896081397309899,0.41390872886404395,\
 0.01432593585923314,0.78838609857484698,\
0.98854076582938433,0.33200952783226967,0.34549844544380903,0.43377211131155491,\
 0.81870661024004221,0.61048321425914764,0.94149570073932409,0.35999277792870998,\
 0.81914897728711367,0.34530424699187279}


#define ZRMATRIX_RESULT {4.6052332809216896,2.16205411267794823,1.12245312965776556,4.0468190853821735,\
 7.29705295628955231,2.45894799152591625,2.29331674633863614,4.21571156291592253,\
 2.81922686558935887,2.348781949769843,\
2.07401297354815162,1.50063692688995753,0.25219828236411757,2.7063265188290884,\
 6.35812780561968260,0.67264506810224134,0.79079814568186024,2.91747981321864280,\
 1.01032692983832129,0.75975015828076931,\
7.04077899650083694,5.39363072316827363,4.99449464719530045,6.79878933472462954,\
 11.2977866849707205,5.7026161792862045,6.09443833294921244,7.35970702338152272,\
 6.14030297304667272,5.80227288585002476,\
1.52422494456658053,0.15792523935350228,-0.39437456229879331,3.21063571532312331,\
 4.72654937095295491,0.19681048033039794,0.13926710536989889,2.4098033377935764,\
 1.19110231590601412,0.24431206393184518,\
5.84910802491279558,3.67968516288822922,2.73879619563976862,5.69547375492497387,\
 10.2284804678743431,4.40378197582331143,3.9855513762320007,6.24858736599414133,\
 4.57057059205936156,4.19331143449899635,\
2.84423960711997825,0.91341486330902766,0.32799575263405639,3.63130361353437925,\
 6.73358898182897025,2.44092642244158631,1.4035651670524776,3.29646069585033841,\
 1.63804041671370615,1.34065096546340268,\
5.04689360696997724,2.58451417966281571,1.89915292921285772,4.62170835433455451,\
 9.05019265678329887,3.33670002112993291,3.66038974983755372,5.62296495186113443,\
 4.04432332701467434,3.48415716881126114,\
2.99051528256943167,0.76077291642727529,0.19995186822386554,2.96972391032047955,\
 6.60462877127714787,1.8346028052798535,1.31621250911818244,3.73816126628208467,\
 1.3788533261260996,0.30973924698106181,\
5.20595600060592734,3.44646883779368096,1.85946748567920750,5.1361216710529769,\
 10.0941765681228333,3.49429925842045552,3.05021742013193098,5.4348094549241184,\
 4.51765668386982977,3.40850797720926035,\
4.65702831667450212,3.03706964016556746,2.06940591125521856,5.11047315752830578,\
 9.94379845012778674,3.3321507049026704,3.29919175160893818,5.94111235727490428,\
 4.22429837741492253,4.41612421969640145}

#define ZIMATRIX_RESULT {-14.2670217568608670,-12.454640354285063,-11.2804810864205063,-12.2724799117031509,\
 -14.2468231668880438,-13.7336716491454212,-13.4126704881317167,-13.2354295209306141,\
 -15.0748008268599314,-16.4106582082002141,\
-18.1383938627890586,-16.2500310962210612,-13.6080240801173602,-16.1094229201551187,\
 -18.8440235801764437,-17.4411014766214336,-16.8096205594578301,-16.4844610002227725,\
 -18.6024277051838922,-19.7565384964549153,\
-13.6088148145822956,-12.746762149308113,-10.9388281270062464,-12.2279241359128505,\
 -13.2072598928012983,-13.2417837783338328,-12.6472077791728861,-12.2778413042734709,\
 -14.8521542656410972,-15.7031049148228732,\
-13.2144981163630746,-11.2231448799035274,-9.7632306445459864,-11.1681914913523084,\
 -13.2326349218083568,-12.3373828754640584,-11.7842111257382829,-11.8870664403988826,\
 -13.3125999502667849,-14.6665812584015534,\
-14.2784445192169489,-13.0211070334450945,-11.1550881344005877,-12.7744646170515193,\
 -13.8646425595775469,-13.6955906712953759,-12.98118927813419,-12.5125330823426655,\
 -15.2470302805889251,-16.6299797243330687,\
-15.3135193060880574,-12.8989039623516373,-11.1097185418698405,-12.9878481756480575,\
 -15.1819592065246791,-14.1554345301917017,-13.7019470831634962,-13.1000905274159241,\
 -15.697504333561957,-16.4727155095115592,\
-16.6151865381969657,-15.0301220873079888,-13.119515101239017,-14.6038902499774128,\
 -16.5465022870658487,-16.2745681097221642,-15.5790325754692791,-14.9778105214080721,\
 -17.5880407064706752,-18.9858728410024717,\
-15.2244202423134851,-12.9437967685452655,-11.4320003274904263,-12.6340981883688794,\
 -15.2648705712828949,-14.0427697860698029,-13.5757724082098221,-13.5147279252801411,\
 -15.9307999370437052,-16.8057661616885916,\
-18.255930649471658,-16.1956269359019345,-13.6756135725252594,-16.3611958886634667,\
 -18.0721805983153097,-17.0517408001458151,-16.231614445430882,-16.4843630200203357,\
 -19.7691884354979948,-20.3871124127432672,\
-18.2105493706955421,-16.9125659016652854,-14.4302205379609418,-16.7267447227195518,\
 -18.7135188739136353,-17.6467699531147559,-16.5820479292089722,-17.0861611432949942,\
 -19.7638794191151916,-21.2748129090505991}

#define FMATRIX_IN  {0.15869047446176410f,0.80895681912079453f,0.72621259605512023f,0.18993748771026731f,\
 0.43964602379128337f,0.37921421090140939f,0.95195201179012656f,0.59145097108557820f,\
 0.70597065938636661f,0.62873698258772492f,\
0.62407150492072105f,0.68317985069006681f,0.70999773032963276f,0.25839814636856318f,\
 0.65407369285821915f,0.76687160786241293f,0.71278580836951733f,0.68067426700145006f,\
 0.70181696489453316f,0.28785153012722731f,\
0.63760355720296502f,0.03401931514963508f,0.47457459988072515f,0.09878737432882190f,\
 0.58781063789501786f,0.60066213225945830f,0.11923700617626309f,0.07392961112782359f,\
 0.40879997471347451f,0.3292048736475408f,\
0.42704886104911566f,0.23805456422269344f,0.9438692079856992f,0.06199027225375175f,\
 0.60208318661898375f,0.78567356057465076f,0.50091631803661585f,0.94336947053670883f,\
 0.06362213846296072f,0.47192330472171307f,\
0.10716815060004592f,0.94920115964487195f,0.14596485672518611f,0.04034968325868249f,\
 0.04535020282492042f,0.73871155502274632f,0.32900535268709064f,0.12863306934013963f,\
 0.06573933875188231f,0.33537696348503232f,\
0.23822965659201145f,0.21827886346727610f,0.07141010463237762f,0.74001471605151892f,\
 0.20294443331658840f,0.55442603398114443f,0.48089468106627464f,0.20190807711333036f,\
 0.53310041315853596f,0.55530697200447321f,\
0.94629473658278584f,0.61546878470107913f,0.67337385797873139f,0.61626600986346602f,\
 0.78442738251760602f,0.99291495559737086f,0.33036959776654840f,0.19693034281954169f,\
 0.03315818728879094f,0.11960808141157031f,\
0.45766852516680956f,0.83135433867573738f,0.65369247179478407f,0.6583583448082209f,\
 0.26375361625105143f,0.97574284672737122f,0.63044753577560186f,0.89286901615560055f,\
 0.31578356493264437f,0.76139996573328972f,\
0.89644787041470408f,0.77340125897899270f,0.19968961318954825f,0.25145971449092031f,\
 0.43832763982936740f,0.37096222722902894f,0.21171907847747207f,0.46179189579561353f,\
 0.37858232436701655f,0.47909884760156274f,\
0.44384705275297165f,0.4244190966710448f,0.60141251794993877f,0.38433500844985247f,\
 0.86648589745163918f,0.30322382133454084f,0.44860231317579746f,0.62512917164713144f,\
 0.46195234358310699f,0.28169692959636450f}

#define FMATRIX_RESULT {14.6458099031118518f,15.9879935334969154f,14.5023146230340352f,9.79104027026660262f,\
 13.658147977601745f,18.2826677794023276f,13.9913488019182726f,13.0555850244756648f,\
 11.2702097163264696f,12.2080852641231488f,\
14.9576657268129818f,18.0415870145381838f,15.4577532112657572f,10.6060207869925502f,\
 14.6987916130120198f,19.9917861348452170f,14.8484467294549827f,14.0994958596124f,\
 12.1294313366477713f,12.8836800651605152f,\
7.72729140087719646f,8.27571528291814396f,8.78222511899259928f,5.22537042408334251f,\
 7.61718988429527233f,10.1106593547443495f,7.22727291477849576f,6.72646688774223733f,\
 6.21430065514425589f,6.69442204331610213f,\
11.8618893865302528f,13.2731189359057637f,12.8373872935372706f,9.52692469532374631f,\
 11.9056327368823247f,16.3654355264039921f,11.8674243780595265f,11.7206072002431814f,\
 9.24712051368910970f,10.6966525674175159f,\
7.25089971320127891f,8.98673751651268127f,7.49831172728240691f,5.29368504368777693f,\
 8.12154956830077346f,10.3089262075282218f,7.4337348502224856f,6.83888206810676191f,\
 5.87125977419815737f,6.5321378683758375f,\
9.25854069422946502f,10.4059421461299539f,9.47515410794143342f,7.263789985253827f,\
 9.16034813421007321f,13.6353552693330933f,9.3726649654333336f,8.83558176180042132f,\
 7.6055387753197277f,8.4871270614444292f,\
12.6878667266786511f,14.1417583108624711f,13.0448499416136254f,9.182764814582713f,\
 12.4852121681423256f,17.073953108953809f,13.2550267101027224f,11.3640159258193041f,\
 9.6279844249894015f,10.6765282334921068f,\
16.0691497196859565f,18.4667994666424704f,16.975871456346809f,12.083598083627443f,\
 15.7522021028722747f,22.0489520587189212f,16.1322911282790358f,16.8044904865300744f,\
 12.7829232118433556f,14.5730443462997599f,\
11.943978463537455f,13.6259118560125358f,11.8068134298221956f,8.21687415305816771f,\
 11.4127920042158735f,15.2296090993128281f,11.3826577511842082f,11.0976684273648587f,\
 10.3857413785092909f,10.2894980707345525f,\
11.5843985401712821f,13.2781579371486753f,12.1725305229437293f,8.37850128011447204f,\
 11.8141382209783163f,15.4022074994757734f,11.484608766021573f,11.1535660239869383f,\
 9.31343988645433285f,11.1349864749012912f}





#define CRMATRIX_IN {0.21132486546412110f,0.56084860628470778f,0.30760907428339124f,0.50153415976092219f, \
 0.28064980218186975f,0.40948254754766822f,0.38737787725403905f,0.53762298030778766f,\
 0.58787201577797532f,0.64885628735646605f,\
0.75604385416954756f,0.66235693730413914f,0.93296162132173777f,0.43685875833034515f,\
 0.12800584640353918f,0.87841258011758327f,0.92228986788541079f,0.11999255046248436f,\
 0.48291792999953032f,0.99231909401714802f,\
0.00022113462910056f,0.72635067673400044f,0.21460078610107303f,0.26931248093023896f,\
 0.77831285959109664f,0.11383596854284406f,0.94881842611357570f,0.2256303490139544f,\
 0.22328650346025825f,0.05004197778180242f,\
0.33032709173858166f,0.19851438421756029f,0.31264199689030647f,0.63257448654621840f,\
 0.21190304495394230f,0.19983377400785685f,0.34353372454643250f,0.62740930821746588f,\
 0.84008856676518917f,0.74855065811425447f,\
0.66538110421970487f,0.54425731627270579f,0.36163610080257058f,0.40519540151581168f,\
 0.11213546665385365f,0.56186607433483005f,0.37601187312975526f,0.76084325974807143f,\
 0.12059959070757031f,0.41040589986369014f,\
0.62839178834110498f,0.23207478970289230f,0.2922266637906432f,0.91847078315913677f,\
 0.68568959552794695f,0.58961773291230202f,0.73409405630081892f,0.04855662025511265f,\
 0.28553641680628061f,0.60845263302326202f,\
0.84974523587152362f,0.23122371966019273f,0.56642488157376647f,0.04373343335464597f,\
 0.15312166837975383f,0.68539796629920602f,0.26157614728435874f,0.67239497276023030f,\
 0.86075146449729800f,0.85442108893766999f,\
0.68573101982474327f,0.21646326314657927f,0.48264719732105732f,0.48185089323669672f,\
 0.69708506017923355f,0.89062247332185507f,0.49934938363730907f,0.20171726960688829f,\
 0.84941016510128975f,0.06426467280834913f,\
0.87821648130193353f,0.88338878145441413f,0.33217189135029912f,0.26395560009405017f,\
 0.84155184263363481f,0.50422128057107329f,0.26385784195736051f,0.39115739194676280f,\
 0.52570608118548989f,0.82790829380974174f,\
0.06837403681129217f,0.65251349471509457f,0.59350947011262178f,0.41481037065386772f,\
 0.40620247554033995f,0.34936154074966908f,0.52535630855709314f,0.83003165572881699f,\
 0.99312098976224661f,0.92623437754809856f}

#define CIMATRIX_IN {0.56672112690284848f,0.73956565884873271f,0.02587099233642220f,0.70648676296696067f,\
 0.96770532103255391f,0.42568723810836673f,0.18711117887869477f,0.21240556007251143f,\
 0.69127879338338971f,0.13049928424879909f,\
0.57116389367729425f,0.00371731072664261f,0.51744682248681784f,0.52114724926650524f,\
 0.50685344357043505f,0.24615605548024178f,0.01895748358219862f,0.57950199581682682f,\
 0.76568587962538004f,0.96820035576820374f,\
0.81601104838773608f,0.59005728596821427f,0.39168732380494475f,0.28704008506610990f,\
 0.52329764096066356f,0.92295324662700295f,0.84335647663101554f,0.26281475694850087f,\
 0.35726496530696750f,0.6561381467618048f,\
0.05689279362559319f,0.3096467358991504f,0.24135384149849415f,0.65027950797230005f,\
 0.5596947595477104f,0.10007458087056875f,0.07485948316752911f,0.43609866220504045f,\
 0.76933998242020607f,0.24455389659851789f,\
0.55959366867318749f,0.25522057106718421f,0.50644348980858922f,0.08813347620889544f,\
 0.56173070007935166f,0.46782181179150939f,0.85328151332214475f,0.91105451015755534f,\
 0.54776339093223214f,0.52831235667690635f,\
0.12493403162807226f,0.62518793903291225f,0.42361020017415285f,0.44987633451819420f,\
 0.46817600261420012f,0.39504976756870747f,0.01245901081711054f,0.80826673656702042f,\
 0.09622887428849936f,0.84689256362617016f,\
0.72792222863063216f,0.11574173765257001f,0.28937275288626552f,0.72272530803456903f,\
 0.77945467224344611f,0.03661171020939946f,0.18675393564626575f,0.81026530498638749f,\
 0.95611717319115996f,0.78766220854595304f,\
0.26777664758265018f,0.61170040909200907f,0.08879321813583374f,0.89767962601035833f,\
 0.79010718129575253f,0.51753685344010592f,0.49205840751528740f,0.25904283951967955f,\
 0.22074085660278797f,0.12620826158672571f,\
0.54653349192813039f,0.67839562846347690f,0.62128817522898316f,0.24278218811377883f,\
 0.98085420625284314f,0.83254515705630183f,0.74896081397309899f,0.41390872886404395f,\
 0.01432593585923314f,0.78838609857484698f,\
0.98854076582938433f,0.33200952783226967f,0.34549844544380903f,0.43377211131155491f,\
 0.81870661024004221f,0.61048321425914764f,0.94149570073932409f,0.35999277792870998f,\
 0.81914897728711367f,0.34530424699187279f}


#define CRMATRIX_RESULT {4.6052332809216896f,2.16205411267794823f,1.12245312965776556f,4.0468190853821735f,\
 7.29705295628955231f,2.45894799152591625f,2.29331674633863614f,4.21571156291592253f,\
 2.81922686558935887f,2.348781949769843f,\
2.07401297354815162f,1.50063692688995753f,0.25219828236411757f,2.7063265188290884f,\
 6.35812780561968260f,0.67264506810224134f,0.79079814568186024f,2.91747981321864280f,\
 1.01032692983832129f,0.75975015828076931f,\
7.04077899650083694f,5.39363072316827363f,4.99449464719530045f,6.79878933472462954f,\
 11.2977866849707205f,5.7026161792862045f,6.09443833294921244f,7.35970702338152272f,\
 6.14030297304667272f,5.80227288585002476f,\
1.52422494456658053f,0.15792523935350228f,-0.39437456229879331f,3.21063571532312331f,\
 4.72654937095295491f,0.19681048033039794f,0.13926710536989889f,2.4098033377935764f,\
 1.19110231590601412f,0.24431206393184518f,\
5.84910802491279558f,3.67968516288822922f,2.73879619563976862f,5.69547375492497387f,\
 10.2284804678743431f,4.40378197582331143f,3.9855513762320007f,6.24858736599414133f,\
 4.57057059205936156f,4.19331143449899635f,\
2.84423960711997825f,0.91341486330902766f,0.32799575263405639f,3.63130361353437925f,\
 6.73358898182897025f,2.44092642244158631f,1.4035651670524776f,3.29646069585033841f,\
 1.63804041671370615f,1.34065096546340268f,\
5.04689360696997724f,2.58451417966281571f,1.89915292921285772f,4.62170835433455451f,\
 9.05019265678329887f,3.33670002112993291f,3.66038974983755372f,5.62296495186113443f,\
 4.04432332701467434f,3.48415716881126114f,\
2.99051528256943167f,0.76077291642727529f,0.19995186822386554f,2.96972391032047955f,\
 6.60462877127714787f,1.8346028052798535f,1.31621250911818244f,3.73816126628208467f,\
 1.3788533261260996f,0.30973924698106181f,\
5.20595600060592734f,3.44646883779368096f,1.85946748567920750f,5.1361216710529769f,\
 10.0941765681228333f,3.49429925842045552f,3.05021742013193098f,5.4348094549241184f,\
 4.51765668386982977f,3.40850797720926035f,\
4.65702831667450212f,3.03706964016556746f,2.06940591125521856f,5.11047315752830578f,\
 9.94379845012778674f,3.3321507049026704f,3.29919175160893818f,5.94111235727490428f,\
 4.22429837741492253f,4.41612421969640145f}

#define CIMATRIX_RESULT {-14.2670217568608670f,-12.454640354285063f,-11.2804810864205063f,-12.2724799117031509f,\
 -14.2468231668880438f,-13.7336716491454212f,-13.4126704881317167f,-13.2354295209306141f,\
 -15.0748008268599314f,-16.4106582082002141f,\
-18.1383938627890586f,-16.2500310962210612f,-13.6080240801173602f,-16.1094229201551187f,\
 -18.8440235801764437f,-17.4411014766214336f,-16.8096205594578301f,-16.4844610002227725f,\
 -18.6024277051838922f,-19.7565384964549153f,\
-13.6088148145822956f,-12.746762149308113f,-10.9388281270062464f,-12.2279241359128505f,\
 -13.2072598928012983f,-13.2417837783338328f,-12.6472077791728861f,-12.2778413042734709f,\
 -14.8521542656410972f,-15.7031049148228732f,\
-13.2144981163630746f,-11.2231448799035274f,-9.7632306445459864f,-11.1681914913523084f,\
 -13.2326349218083568f,-12.3373828754640584f,-11.7842111257382829f,-11.8870664403988826f,\
 -13.3125999502667849f,-14.6665812584015534f,\
-14.2784445192169489f,-13.0211070334450945f,-11.1550881344005877f,-12.7744646170515193f,\
 -13.8646425595775469f,-13.6955906712953759f,-12.98118927813419f,-12.5125330823426655f,\
 -15.2470302805889251f,-16.6299797243330687f,\
-15.3135193060880574f,-12.8989039623516373f,-11.1097185418698405f,-12.9878481756480575f,\
 -15.1819592065246791f,-14.1554345301917017f,-13.7019470831634962f,-13.1000905274159241f,\
 -15.697504333561957f,-16.4727155095115592f,\
-16.6151865381969657f,-15.0301220873079888f,-13.119515101239017f,-14.6038902499774128f,\
 -16.5465022870658487f,-16.2745681097221642f,-15.5790325754692791f,-14.9778105214080721f,\
 -17.5880407064706752f,-18.9858728410024717f,\
-15.2244202423134851f,-12.9437967685452655f,-11.4320003274904263f,-12.6340981883688794f,\
 -15.2648705712828949f,-14.0427697860698029f,-13.5757724082098221f,-13.5147279252801411f,\
 -15.9307999370437052f,-16.8057661616885916f,\
-18.255930649471658f,-16.1956269359019345f,-13.6756135725252594f,-16.3611958886634667f,\
 -18.0721805983153097f,-17.0517408001458151f,-16.231614445430882f,-16.4843630200203357f,\
 -19.7691884354979948f,-20.3871124127432672f,\
-18.2105493706955421f,-16.9125659016652854f,-14.4302205379609418f,-16.7267447227195518f,\
 -18.7135188739136353f,-17.6467699531147559f,-16.5820479292089722f,-17.0861611432949942f,\
 -19.7638794191151916f,-21.2748129090505991f}

/* FIXME: assert are 1e04 and 1e14 */
static void sexpmaTest (void ) {
  
  int i = 0 ;
    
  float in[] = FMATRIX_IN ;
    
  float result[] = FMATRIX_RESULT ;
  
  float out[LEADDIM*LEADDIM] ;
    
  float mon_test[9]={4,8,3,2,8,4,3,4,5};
  float result_mon_test[9]={179660.90104359941324219f,452187.79172967246267945f,273148.88425553269917145f,
  163436.47016010177321732f,411353.63271151197841391f,248489.23907714022789150f,
  138776.82498170930193737f,349268.28353250800864771f,210989.61389596699154936f}; 
  float out_mon_test[9] ;
    
  sexpma ( in,  out, LEADDIM) ; 
        
  	for ( i = 0 ; i < LEADDIM*LEADDIM  ; i++ )
	{
		printf ( "\t\t %d out : %e\tresult : %e\tassert : %e \n" , i , out[i] , result[i] , fabs ( out[i] - result[i] ) / fabs( out[i]) ) ;	
		assert ( fabs ( out[i] - result[i] ) / fabs( out[i]) < 3e-6 ) ;
	} 
	
  sexpma(mon_test,out_mon_test,3);
    	for ( i = 0 ; i < 9  ; i++ )
	{
		printf ( "\t\t %d out : %e\tresult : %e\tassert : %e \n" , i , out[i] , result[i] , fabs ( out[i] - result[i] ) / fabs( out[i]) ) ;	
		assert ( fabs ( out_mon_test[i] - result_mon_test[i] ) / fabs( out_mon_test[i]) < 3e-5 ) ;
	}  
}



static void dexpmaTest (void ) {
  
  int i = 0 ;
    
  double in[] = DMATRIX_IN ;
    
  double result[] = DMATRIX_RESULT ;
  
  double out[LEADDIM*LEADDIM] ;
    
    
  dexpma ( in,  out, LEADDIM) ; 
        
  	for ( i = 0 ; i < LEADDIM*LEADDIM  ; i++ )
	{
		printf ( "\t\t %d out : %e\tresult : %e\tassert : %e \n" , i , out[i] , result[i] , fabs ( out[i] - result[i] ) / fabs( out[i]) ) ;	
		assert ( fabs ( out[i] - result[i] ) / fabs( out[i]) < 3e-14 ) ;
	}  
}


static void cexpmaTest ( void) {
    
    int i = 0 ; 
    
    float tRealIn [] = CRMATRIX_IN ;
    float tImagIn [] = CIMATRIX_IN ;
    

        
    float tRealResult [] = CRMATRIX_RESULT ;
    float tImagResult [] = CIMATRIX_RESULT ;
    
    floatComplex  out[LEADDIM*LEADDIM ] ;
    
    floatComplex* in       = FloatComplexMatrix ( tRealIn , tImagIn , LEADDIM*LEADDIM );    
    floatComplex* Result   = FloatComplexMatrix ( tRealResult , tImagResult ,LEADDIM*LEADDIM) ;
    
    cexpma ( in ,out , LEADDIM ) ;
    
    
        /* if we don't add that test assert failed if result = 0  'cause then we have  |(out - 0)|/|out| = 1*/     
	for ( i = 0 ; i < (LEADDIM*LEADDIM )  ; i++ )
	{
	  printf ( "\t\t %d out : %e\t + %e\t * i  result : %e\t + %e\t * i  assert : %e + %e \n" ,
              i ,creals(out[i]) , cimags(out[i]) , creals (Result[i])  , cimags (Result[i]),
              fabs(  creals(out[i]) -  creals (Result[i]) ) / fabs (creals (out[i])) ,
              fabs(  cimags(out[i]) -  cimags (Result[i]) ) / fabs (cimags (out[i])));
           
    if (  creals(out[i])  < 3e-16 && creals (Result[i]) < 1e-18 )
        assert ( 1 ) ;
    else         
        assert ( fabs(  creals(out[i]) -  creals (Result[i]) ) / fabs (creals (out[i]))  < 3e-4 );
    
        
    if (  cimags(out[i])  < 3e-16 && cimags (Result[i]) < 1e-18 )
        assert ( 1 ) ;
    else         
	    assert ( fabs(  cimags(out[i]) -  cimags (Result[i]) ) / fabs (cimags (out[i]))  < 3e-6 ) ;
    
    }
}



static void zexpmaTest ( void) {
    
    int i = 0 ; 
    
    double tRealIn [] = ZRMATRIX_IN ;
    double tImagIn [] = ZIMATRIX_IN ;
    

        
    double tRealResult [] = ZRMATRIX_RESULT ;
    double tImagResult [] = ZIMATRIX_RESULT ;
    
    doubleComplex  out[LEADDIM*LEADDIM ] ;
    
    doubleComplex* in       = DoubleComplexMatrix ( tRealIn , tImagIn , LEADDIM*LEADDIM );    
    doubleComplex* Result   = DoubleComplexMatrix ( tRealResult , tImagResult ,LEADDIM*LEADDIM) ;
    
    zexpma ( in ,out , LEADDIM ) ;
    
    
        /* if we don't add that test assert failed if result = 0  'cause then we have  |(out - 0)|/|out| = 1*/     
	for ( i = 0 ; i < (LEADDIM*LEADDIM )  ; i++ )
	{
	  printf ( "\t\t %d out : %e\t + %e\t * i  result : %e\t + %e\t * i  assert : %e + %e \n" ,
              i ,zreals(out[i]) , zimags(out[i]) , zreals (Result[i])  , zimags (Result[i]),
              fabs(  zreals(out[i]) -  zreals (Result[i]) ) / fabs (zreals (out[i])) ,
              fabs(  zimags(out[i]) -  zimags (Result[i]) ) / fabs (zimags (out[i])));
           
    if (  zreals(out[i])  < 3e-16 && zreals (Result[i]) < 1e-18 )
        assert ( 1 ) ;
    else         
        assert ( fabs(  zreals(out[i]) -  zreals (Result[i]) ) / fabs (zreals (out[i]))  < 3e-13 );
    
        
    if (  zimags(out[i])  < 3e-16 && zimags (Result[i]) < 1e-18 )
        assert ( 1 ) ;
    else         
	    assert ( fabs(  zimags(out[i]) -  zimags (Result[i]) ) / fabs (zimags (out[i]))  < 3e-14) ;
    
    }
}


static int testExponential(void) {

  printf("\n>>>> Matrix Exponential Tests\n");
  printf("\t>>>> Matrix Double Realt Tests\n");  
  dexpmaTest();

  printf("\n\n\t>>>> Matrix Float Realt Tests\n");     
  sexpmaTest();
  
  printf("\n\n\n");
  printf("\t>>>> Matrix Float Complex Tests\n");      
  cexpmaTest();  
    
  printf("\n\n\n");
  printf("\t>>>> Matrix Double Complex Tests\n");  
  zexpmaTest();

  return 0;
}



int main(void) {
  assert(testExponential() == 0);
  return 0;
}
